document.addEventListener('DOMContentLoaded', function() {
    // Elementos do DOM
    const form = document.getElementById('formulario-pier');
    const steps = document.querySelectorAll('.form-step');
    const progressBar = document.getElementById('progress');
    const stepIndicators = document.querySelectorAll('.step');
    const nextButtons = document.querySelectorAll('.next-btn');
    const prevButtons = document.querySelectorAll('.prev-btn');
    const successMessage = document.getElementById('success-message');
    const errorMessage = document.getElementById('error-message');
    const tryAgainButton = document.getElementById('try-again');
    const protocoloSpan = document.getElementById('protocolo');
    
    // Criar elemento de loading
    const loadingIndicator = document.createElement('div');
    loadingIndicator.className = 'loading-indicator hidden';
    loadingIndicator.innerHTML = `
        <div class="spinner"></div>
        <p>Enviando dados...</p>
        <p class="status-message"></p>
    `;
    document.querySelector('.container').appendChild(loadingIndicator);
    const statusMessage = loadingIndicator.querySelector('.status-message');

    // Variáveis de controle
    let currentStep = 0;
    let authToken = null;
    let tokenExpiration = null;

    // Configurações - APONTANDO PARA OS ENDPOINTS PHP NO SEU SERVIDOR
    const config = {
        apiBaseUrl: 'https://contratar.simplescon.com/abrir/api', // Caminho para os scripts PHP
        tokenValidityMinutes: 60
    };

    // Adicionar estilos dinâmicos
    const style = document.createElement('style');
    style.textContent = `
        .loading-indicator {
            position: fixed;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background-color: rgba(255, 255, 255, 0.9);
            display: flex;
            flex-direction: column;
            justify-content: center;
            align-items: center;
            z-index: 1000;
            font-size: 1.2rem;
        }
        .loading-indicator.hidden {
            display: none;
        }
        .spinner {
            border: 5px solid #f3f3f3;
            border-top: 5px solid var(--primary-color);
            border-radius: 50%;
            width: 50px;
            height: 50px;
            animation: spin 1s linear infinite;
            margin-bottom: 20px;
        }
        .status-message {
            margin-top: 15px;
            color: var(--dark-color);
            font-style: italic;
        }
        @keyframes spin {
            0% { transform: rotate(0deg); }
            100% { transform: rotate(360deg); }
        }
        .error-field {
            border-color: var(--error-color) !important;
            box-shadow: 0 0 0 2px rgba(231, 76, 60, 0.2);
        }
    `;
    document.head.appendChild(style);

    // Inicialização
    updateProgress();

    // Máscaras para campos
    const cpfInput = document.getElementById('cpf');
    if (cpfInput) {
        cpfInput.addEventListener('input', maskCPF);
    }

    const telefoneInput = document.getElementById('telefone');
    if (telefoneInput) {
        telefoneInput.addEventListener('input', maskTelefone);
    }

    const cepInput = document.getElementById('cep');
    if (cepInput) {
        cepInput.addEventListener('input', maskCEP);
        cepInput.addEventListener('blur', buscarCEP);
    }

    const capitalInput = document.getElementById('capital_social');
    if (capitalInput) {
        capitalInput.addEventListener('input', maskCapital);
    }

    // Navegação entre etapas
    nextButtons.forEach(button => {
        button.addEventListener('click', nextStep);
    });

    prevButtons.forEach(button => {
        button.addEventListener('click', prevStep);
    });

    // Envio do formulário
    form.addEventListener('submit', handleSubmit);

    // Tentar novamente em caso de erro
    if (tryAgainButton) {
        tryAgainButton.addEventListener('click', resetForm);
    }

    // Funções de máscara
    function maskCPF(e) {
        let value = e.target.value.replace(/\D/g, '');
        if (value.length > 11) value = value.slice(0, 11);
        
        if (value.length > 9) {
            value = value.replace(/^(\d{3})(\d{3})(\d{3})(\d{2})$/, '$1.$2.$3-$4');
        } else if (value.length > 6) {
            value = value.replace(/^(\d{3})(\d{3})(\d{1,3})/, '$1.$2.$3');
        } else if (value.length > 3) {
            value = value.replace(/^(\d{3})(\d{1,3})/, '$1.$2');
        }
        
        e.target.value = value;
    }

    function maskTelefone(e) {
        let value = e.target.value.replace(/\D/g, '');
        if (value.length > 11) value = value.slice(0, 11);
        
        if (value.length > 10) {
            value = value.replace(/^(\d{2})(\d{5})(\d{4})$/, '($1) $2-$3');
        } else if (value.length > 6) {
            value = value.replace(/^(\d{2})(\d{4})(\d{0,4})/, '($1) $2-$3');
        } else if (value.length > 2) {
            value = value.replace(/^(\d{2})(\d{0,5})/, '($1) $2');
        }
        
        e.target.value = value;
    }

    function maskCEP(e) {
        let value = e.target.value.replace(/\D/g, '');
        if (value.length > 8) value = value.slice(0, 8);
        
        if (value.length > 5) {
            value = value.replace(/^(\d{5})(\d{1,3})/, '$1-$2');
        }
        
        e.target.value = value;
    }

    function maskCapital(e) {
        let value = e.target.value.replace(/\D/g, '');
        value = (parseInt(value || '0') / 100).toLocaleString('pt-BR', {
            minimumFractionDigits: 2,
            maximumFractionDigits: 2
        });
        e.target.value = value;
    }

    // Busca de CEP
    async function buscarCEP() {
        const cep = this.value.replace(/\D/g, '');
        if (cep.length !== 8) return;

        try {
            statusMessage.textContent = 'Buscando endereço...';
            loadingIndicator.classList.remove('hidden');
            
            const response = await fetch(`https://viacep.com.br/ws/${cep}/json/`);
            const data = await response.json();

            if (data.erro) {
                throw new Error('CEP não encontrado');
            }

            document.getElementById('logradouro').value = data.logradouro;
            document.getElementById('bairro').value = data.bairro;
            document.getElementById('cidade').value = data.localidade;
            document.getElementById('estado').value = data.uf;
            document.getElementById('numero').focus();
        } catch (error) {
            console.error('Erro ao buscar CEP:', error);
            showError('CEP não encontrado. Preencha manualmente.');
        } finally {
            loadingIndicator.classList.add('hidden');
            statusMessage.textContent = '';
        }
    }

    // Navegação entre etapas
    function nextStep() {
        if (validateStep(currentStep)) {
            currentStep++;
            updateStep();
            updateProgress();
        }
    }

    function prevStep() {
        currentStep--;
        updateStep();
        updateProgress();
    }

    // Validação de etapa
    function validateStep(stepIndex) {
        const currentStepElement = steps[stepIndex];
        const inputs = currentStepElement.querySelectorAll('input[required], select[required], textarea[required]');
        let isValid = true;
        
        inputs.forEach(input => {
            if (!input.value.trim()) {
                isValid = false;
                input.classList.add('error-field');
                
                input.addEventListener('input', function() {
                    this.classList.remove('error-field');
                }, { once: true });
            }
        });
        
        if (!isValid) {
            showError('Preencha todos os campos obrigatórios');
            currentStepElement.scrollIntoView({ behavior: 'smooth', block: 'start' });
        }
        
        return isValid;
    }

    // Atualizar visualização
    function updateStep() {
        steps.forEach((step, index) => {
            step.classList.remove('active');
            if (index === currentStep) {
                step.classList.add('active');
            }
        });
    }

    function updateProgress() {
        const progress = (currentStep / (steps.length - 1)) * 100;
        progressBar.style.width = `${progress}%`;
        
        stepIndicators.forEach((step, index) => {
            step.classList.remove('active', 'completed');
            
            if (index < currentStep) {
                step.classList.add('completed');
            } else if (index === currentStep) {
                step.classList.add('active');
            }
        });
    }

    // Obter token de autenticação via PHP proxy
    async function obterToken() {
        if (authToken && tokenExpiration && new Date() < tokenExpiration) {
            return authToken;
        }

        try {
            statusMessage.textContent = 'Autenticando...';
            loadingIndicator.classList.remove('hidden');
            
            const response = await fetch(`${config.apiBaseUrl}/login.php`, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                }
            });

            if (!response.ok) {
                throw new Error(`Erro HTTP: ${response.status}`);
            }

            const authData = await response.json();

            if (!authData.ok || !authData.token) {
                throw new Error('Token não recebido');
            }

            authToken = authData.token;
            const expirationDate = new Date();
            expirationDate.setMinutes(expirationDate.getMinutes() + config.tokenValidityMinutes);
            tokenExpiration = expirationDate;

            return authToken;
        } catch (error) {
            console.error('Erro ao obter token:', error);
            throw error;
        } finally {
            statusMessage.textContent = '';
        }
    }

    // Enviar dados para API via PHP proxy
    async function enviarParaAPIPier(data) {
    const submitButton = form.querySelector('.submit-btn');
    submitButton.disabled = true;
    submitButton.textContent = 'Enviando...';
    loadingIndicator.classList.remove('hidden');
    form.classList.add('hidden');

    try {
        const token = await obterToken();
        statusMessage.textContent = 'Enviando dados...';
        
        // Converter os dados em parâmetros GET
        const queryParams = new URLSearchParams();
        for (const key in data) {
            queryParams.append(key, data[key]);
        }

        const response = await fetch(`${config.apiBaseUrl}/solicitacao.php?${queryParams}`, {
            method: 'GET', // Alterado para GET
            headers: {
                'Authorization': `Bearer ${token}`,
                'Accept': 'application/json'
            }
        });

        if (!response.ok) {
            throw new Error(`Erro HTTP: ${response.status}`);
        }

        const responseData = await response.json();

        if (!responseData.success) {
            throw new Error(responseData.message || 'Erro ao processar solicitação');
        }

        successMessage.classList.remove('hidden');
        protocoloSpan.textContent = responseData.protocolo || 'N/A';
    } catch (error) {
        console.error('Erro no processo:', error);
        showError(error.message || 'Erro ao enviar formulário. Tente novamente.');
    } finally {
        loadingIndicator.classList.add('hidden');
        submitButton.disabled = false;
        submitButton.textContent = 'Enviar';
        statusMessage.textContent = '';
    }
}

    // Manipulador de envio
    async function handleSubmit(e) {
    e.preventDefault();
    
    if (!validateStep(currentStep)) return;
    
    const formData = new FormData(form);
    const formDataObj = Object.fromEntries(formData.entries());

    // Dados no formato exigido pela API PIER
    const dadosParaAPI = {
        cliente_id: "00122175",
        tiposolicitacao_id: "0010b5bd",
        responsavel_id: "000fba73",
        assunto: "Nova Abertura de Empresa pelo Site",
        observacoes: "Dados preenchido pelo leed no site",
        // Inclui todos os dados do formulário
        ...formDataObj
    };

    console.log("Dados sendo enviados para PIER API:", dadosParaAPI);
    
    await enviarParaAPIPier(dadosParaAPI);
}


function gerarTextoFormatado(data) {
    // Formatação mais simples para GET
    return `Nome: ${data.nome}|CPF: ${data.cpf}|Email: ${data.email}|Telefone: ${data.telefone}|` +
           `CEP: ${data.cep}|Endereço: ${data.logradouro}, ${data.numero}|` +
           `Cidade: ${data.cidade}|Estado: ${data.estado}|` +
           `Empresa: ${data.nome_empresa}|Tipo: ${data.tipo_empresa}|` +
           `CNAE: ${data.cnae_principal}|Capital: ${data.capital_social}|` +
           `Atividades: ${data.descricao_atividades.substring(0, 200)}`;
}

    // Mostrar mensagens de erro
    function showError(message) {
        errorMessage.querySelector('p').textContent = message;
        errorMessage.classList.remove('hidden');
        setTimeout(() => errorMessage.scrollIntoView({ behavior: 'smooth' }), 100);
    }

    // Resetar formulário
    function resetForm() {
        errorMessage.classList.add('hidden');
        form.classList.remove('hidden');
        currentStep = 0;
        updateStep();
        updateProgress();
    }
});